
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Sun Jul 22 10:17:10 2007
   *  Last modified   :  %modify_time%
   */
  /** @file 
   *  \brief A brief description of this module 
   *
   *  \version KMNG_ATP_TST_ManageOp.c#1:csrc:1
   *  \author NogaD
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************* Include Files ****************/
#include "DX_VOS_BaseTypes.h"
#include "DX_VOS_Mem.h"
#include "KMNG_Defs.h"
#include "KMNG_API.h"
#include "KMNG_ATP_TST_Defines.h"
#include "KMNG_ATP_TST_ManageOp.h"
#include "KMNG_ATP_TST_KeysOp.h"
#include "KMNG_TST_Utils.h"

/************************ Defines *******************************/
/************************ Enums *********************************/
/************************ Typedefs ******************************/
/************************ Global Data ***************************/
extern const DxUint32_t TST_NumOf_ManageOpTests;
extern const DxUint32_t TST_NumOf_SymListTests;
extern const DxUint32_t TST_NumOf_RSAListTests;
extern const DxUint32_t TST_NumOf_DHListTests;
extern const DxUint32_t TST_NumOf_KeysRSADataTests;
extern const DxUint32_t TST_NumOf_KeysSymDataTests;


extern TST_RingManage_TestDataStruct TST_RingManage_TestData[];
extern TST_RingListDH_TestDataStruct TST_RingListDH_TestData[];
extern TST_RingListRSA_TestDataStruct TST_RingListRSA_TestData[];
extern TST_RingListSym_TestDataStruct TST_RingListSym_TestData[];
extern TST_KeysSym_TestDataStruct TST_TwoRingsSymKey_TestData[];
extern TST_KeysRSA_TestDataStruct TST_TwoRingsRSAKey_TestData[];
extern TST_KeysSym_TestDataStruct TST_KeysSymData_TestData[];
extern TST_KeysRSA_TestDataStruct TST_KeysRSAData_TestData[];

/* A buffer to contain the key ring */
DxUint32_t TST_KeyRing_Buf[KMNG_TST_MAX_SIZE_KEY_RING_BUF];
/* A buffer to simulate flash */
DxUint32_t TST_KeyRing_FlashBuf[KMNG_TST_MAX_SIZE_KEY_RING_BUF];

/************* Private function prototype ***********************/
DxError_t KMNG_TST_Key_Ring_Initialization(DxUint32_t NumOfSymKeys, DxUint32_t NumOfRSAKeys, 
                                           DxUint32_t NumOfDHKeys, DxUint8_t* KeyRingPwd, 
                                           DxUint32_t PwdLen, DxUint8_t* KeyRing_ptr, 
                                           DxUint32_t KeyRingBufferSize);
DxError_t KMNG_TST_List_Key_IDs(DxUint32_t NumOfSymKeys, DxUint32_t NumOfRSAKeys, 
                                DxUint32_t NumOfDHKeys, DxUint32_t* KeyIdsList,
                                DxUint8_t ListOffset, DxUint8_t* KeyRing_ptr,
                                DxUint8_t sortList);
DxError_t KMNG_TST_Retrieve_KeysData_And_Compare(TST_KeysRSA_TestDataStruct *RSAKey, 
                                                 TST_KeysSym_TestDataStruct *SYMKey);
/************************ Public Functions ******************************/
/*****************************************************************************
 * Function Name:                                                           
 *  KMNG_ATP_TST_Key_Ring_Management      
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is an acceptance test for Key management API's. The function 
 *  initializes a key ring according to requested num of keys
 *  
 * Algorithm:       
 * 1. Get key ring buffer size for X keys (symmetric, RSA and DH)
 * 2. Allocate a buffer for the key ring according to the needed size 
 * 3. Initialize a key ring
 * 4. Get key ring capacity and compare to expected size
 * 5. Get key ring current size (0 objects) and compare to expected size
 *
 *  * Implements KMNG ATP section 5.1 
 *
 ******************************************************************************/                                
DxError_t KMNG_ATP_TST_Key_Ring_Management( void ) 
{

    /********* TST Variables Definitions ************/
    DxUint32_t            TST_index;    
    DxError_t             TST_Status = DX_SUCCESS;        
    DxChar                TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_ATP_TST_Key_Ring_Management"; 
    DxUint32_t            TST_RequiredBufSize;
    DxUint8_t             *TST_KeyRingPtr = DX_NULL;
    DxUint32_t            TST_ReceivedNumSymKeys;
    DxUint32_t            TST_ReceivedNumRSAKeys;
    DxUint32_t            TST_ReceivedNumDHKeys;
    
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"******** KMNG_ATP_TST_Key_Ring_Management Test* ********************* \n"));      
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n \n"));

    /************ Main Loop **********/
    for (TST_index = 0; TST_index < TST_NumOf_ManageOpTests; TST_index++)
    {
      
	  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
	  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG," %s \n",TST_RingManage_TestData[TST_index].TST_Name));      
	  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n \n"));	
	  /* Get key ring expected buffer size */
      TST_Status = KMNG_GetKeyRingBufferSize(TST_RingManage_TestData[TST_index].TST_NumSymKeys, 
                                             TST_RingManage_TestData[TST_index].TST_NumRSAKeys,
                                             TST_RingManage_TestData[TST_index].TST_NumDHKeys,
                                             &TST_RequiredBufSize); 
      
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                 "KMNG_GetKeyRingBufferSize",
                                 TST_RingManage_TestData[TST_index].TST_Name,
                                 TST_MasterTestName);
                                                 

      
      if (TST_RequiredBufSize == 0)
      {
        ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"KMNG_GetKeyRingBufferSize returned buffer size of 0 \n")); 
        return DX_FAILURE; 
      }
      
      /* Allocate buffer for the Key Ring */
      TST_KeyRingPtr = (DxUint8_t*)DX_VOS_MemMalloc(TST_RequiredBufSize*sizeof(DxUint8_t));
      if (TST_KeyRingPtr == DX_NULL)
      {
        ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Failed to allocate buffer of size - %d\n",TST_RequiredBufSize));   
        return DX_FAILURE;
      }
      
      /* Initialize Key Ring */
      TST_Status = KMNG_KeyRingInit(TST_RingManage_TestData[TST_index].TST_PwdLen,
                                    TST_RingManage_TestData[TST_index].TST_Pwd,
                                    TST_RingManage_TestData[TST_index].TST_NumSymKeys,
                                    TST_RingManage_TestData[TST_index].TST_NumRSAKeys,
                                    TST_RingManage_TestData[TST_index].TST_NumDHKeys, 
                                    TST_KeyRingPtr);
                                    
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                 "KMNG_KeyRingInit",
                                 TST_RingManage_TestData[TST_index].TST_Name,
                                 TST_MasterTestName);
                                                 
                                  
      
      /* Get Key Ring capacity*/
      TST_Status = KMNG_GetKeyRingCapacity(&TST_ReceivedNumSymKeys, 
                                           &TST_ReceivedNumRSAKeys,
                                           &TST_ReceivedNumDHKeys,
                                           TST_KeyRingPtr);                                           
                                                 
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                 "KMNG_GetKeyRingCapacity",
                                 TST_RingManage_TestData[TST_index].TST_Name,
                                 TST_MasterTestName);
                                                 

      
      /* Check if received number of keys is as expected */
      TST_Status = KMNG_TST_Check_Num_Of_Keys(TST_ReceivedNumSymKeys, TST_RingManage_TestData[TST_index].TST_NumSymKeys,
                                              TST_ReceivedNumRSAKeys, TST_RingManage_TestData[TST_index].TST_NumRSAKeys,
                                              TST_ReceivedNumDHKeys, TST_RingManage_TestData[TST_index].TST_NumDHKeys);
      if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;                                              
      
      /* Get the Key Ring current size */
      TST_Status = KMNG_GetKeyRingSize(&TST_ReceivedNumSymKeys, 
                                       &TST_ReceivedNumRSAKeys,
                                       &TST_ReceivedNumDHKeys,
                                       TST_KeyRingPtr);
      
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                 "KMNG_GetKeyRingSize",
                                 TST_RingManage_TestData[TST_index].TST_Name,
                                 TST_MasterTestName);
                                                 

      
      /* Check if received number of keys is as expected */
      TST_Status = KMNG_TST_Check_Num_Of_Keys(TST_ReceivedNumSymKeys, 0,
                                              TST_ReceivedNumRSAKeys, 0,
                                              TST_ReceivedNumDHKeys, 0);
      if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;
      
      /* Free the allocated buffer for the Key Ring */
      DX_VOS_MemFree(TST_KeyRingPtr);
    }
    
    
        return DX_SUCCESS;

EXIT_ON_ERROR:
    {
      return DX_FAILURE;
    }
}/* End of KMNG_ATP_TST_Key_Ring_Management*/ 


/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/


/*****************************************************************************
 * Function Name:                                                           
 *  KMNG_ATP_TST_Key_Ring_List      
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is an acceptance test for Key management API's. The function 
 *  tests the list key IDs function.
 *  
 * Algorithm:       
 * 1. Initialize a key ring according to requested number of keys
 * 2. Import Symmetric,RSA and DH keys
 * 3. Get the key ring current size, verify its correct and list all the key IDs
 *    Verify that the Key IDs are as expected
 * 4. Export all the imported keys, verify that the exported keys are as expected
 * 5. Delete a symmetric RSA and DH key (in a loop till all keys are deleted)
 * 6. List the keys after each deletion, verify that the remained keys are as 
 *    expected
 * 
 *  * Implements KMNG ATP section 5.5 
 *
 ******************************************************************************/                                
DxError_t KMNG_ATP_TST_Key_Ring_List( void ) 
{

    /********* TST Variables Definitions ************/
    DxUint32_t            TST_index, TST_tempInd=0;    
    DxError_t             TST_Status = DX_SUCCESS;        
    DxChar                TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_ATP_TST_Key_Ring_List"; 
    DxUint8_t             TST_KeyRingPwd[TST_KMNG_PWD_MAX_LEN] = {0x0,0x1,0x2,0x3,0x4,0x5,0x6,0x7,0x8,0x9};
    DxUint32_t            TST_KeyRingPwdLen = 10;
    DxUint32_t            TST_KeyIds[TST_KMNG_MAX_NUM_OF_KEYS];               
    KMNG_KeyType_t        TST_ExportedkeyType;
    DxUint32_t            TST_ExportedSize;
    DxUint32_t            TST_ExportedkeyUsage;
    DxUint32_t            TST_ExportedkeyRestriction;
                              
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"************** KMNG_ATP_TST_Key_Ring_List Test********************** \n"));      
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n \n"));
    
    /************ Initialize Key Ring ************/
    TST_Status = KMNG_TST_Key_Ring_Initialization(TST_RingManage_TestData[1].TST_NumSymKeys, TST_RingManage_TestData[1].TST_NumRSAKeys, 
                                                  TST_RingManage_TestData[1].TST_NumDHKeys, TST_KeyRingPwd, 
                                                  TST_KeyRingPwdLen,(DxUint8_t*)TST_KeyRing_Buf, KMNG_TST_MAX_SIZE_KEY_RING_BUF_BYTES);
                                           
    if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;                                       
                                           
    /************ Import keys Loop **********/
    /* the loop is according to the sym keys,in this test 
       number of sym keys = num of RSA keys = num of DH keys */    
    for (TST_index = 0; TST_index < TST_NumOf_SymListTests; TST_index++)
    {      
      /* Import Sym key */
   	  TST_Status = KMNG_ImportSymUserKey(TST_KeyRingPwd,
                                         TST_KeyRingPwdLen,
                                         TST_RingListSym_TestData[TST_index].TST_KeyPwd, 
                                         TST_RingListSym_TestData[TST_index].TST_KeyPwdLen, 
                                         TST_RingListSym_TestData[TST_index].TST_KeyType,
                                         TST_RingListSym_TestData[TST_index].TST_KeySize,
                                         TST_RingListSym_TestData[TST_index].TST_KeyUsage, 
                                         TST_RingListSym_TestData[TST_index].TST_KeyRestriction,
                                         DX_NULL,
                                         TST_RingListSym_TestData[TST_index].TST_key,
                                         &TST_KeyIds[TST_tempInd],
                                         (DxUint8_t*)TST_KeyRing_Buf); 
      
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                "KMNG_ImportSymUserKey",
                                TST_RingListSym_TestData[TST_index].TST_Name,
                                TST_MasterTestName);
                                                         

      
      TST_tempInd++;
      
      /* Import RSA key */
      TST_Status = KMNG_ImportRsaUserKey(TST_KeyRingPwd, 
                                         TST_KeyRingPwdLen,
                                         TST_RingListRSA_TestData[TST_index].TST_KeyPwd ,
                                         TST_RingListRSA_TestData[TST_index].TST_KeyPwdLen,
                                         TST_RingListRSA_TestData[TST_index].TST_KeyType,
                                         TST_RingListRSA_TestData[TST_index].TST_KeySize,
                                         TST_RingListRSA_TestData[TST_index].TST_KeyUsage,
                                         TST_RingListRSA_TestData[TST_index].TST_KeyRestriction,
                                         DX_NULL,
                                         TST_RingListRSA_TestData[TST_index].TST_N,
                                         TST_RingListRSA_TestData[TST_index].TST_NSize,
                                         TST_RingListRSA_TestData[TST_index].TST_E,
                                         TST_RingListRSA_TestData[TST_index].TST_ESize,
                                         TST_RingListRSA_TestData[TST_index].TST_D,
                                         TST_RingListRSA_TestData[TST_index].TST_DSize,
                                         &TST_KeyIds[TST_tempInd],
                                         (DxUint8_t*)TST_KeyRing_Buf);
                                         
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
												        "KMNG_ImportRsaUserKey",
												        TST_RingListRSA_TestData[TST_index].TST_Name,
												        TST_MasterTestName);
                                                 
            
      TST_tempInd++;                                        
                                              
    } /*End of for loop*/
    
    /************ Get Key Ring size & List Key IDs **********/
    TST_Status = KMNG_TST_List_Key_IDs(TST_NumOf_SymListTests, TST_NumOf_RSAListTests, 
                                       0, TST_KeyIds,0,
                                       (DxUint8_t*)TST_KeyRing_Buf, TST_KMNG_SORT_LIST_A);
    
    if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;                                       
    
    /************ Export keys Loop **********/
    /* the loop is according to the sym keys,in this test 
       number of sym keys = num of RSA keys = num of DH keys */    
    TST_tempInd = 0;
    for (TST_index = 0; TST_index < TST_NumOf_SymListTests; TST_index++)
    {            
      /* Export Sym key */      
	  TST_Status = KMNG_ExportSymUserKey(TST_KeyRingPwd, 
                                         TST_KeyRingPwdLen,
                                         TST_RingListSym_TestData[TST_index].TST_KeyPwd, 
                                         TST_RingListSym_TestData[TST_index].TST_KeyPwdLen, 
                                         TST_KeyIds[TST_tempInd],
                                         TST_RingListSym_TestData[TST_index].TST_Exportedkey,
                                         &TST_ExportedkeyType,
                                         &TST_ExportedSize,
                                         &TST_ExportedkeyUsage,
                                         &TST_ExportedkeyRestriction,
                                         (DxUint8_t*)TST_KeyRing_Buf);
                                         
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
													      "KMNG_ExportSymUserKey",
													      TST_RingListSym_TestData[TST_index].TST_Name,
													      TST_MasterTestName);                                              
      
      /* Compare exported key to imported key key type & key*/
      if (TST_ExportedkeyType != TST_RingListSym_TestData[TST_index].TST_KeyType)
      {
        ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Exported symmetric key type is different from imported symmetric key type\n"));
        return DX_FAILURE;    
      }
      TST_Status = (DxUint32_t)DX_VOS_MemCmp(TST_RingListSym_TestData[TST_index].TST_Exportedkey,TST_RingListSym_TestData[TST_index].TST_key, 
                                 sizeof(TST_RingListSym_TestData[TST_index].TST_key));
      if (TST_Status != DX_SUCCESS)
      {
        ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Exported symmetric key is different from imported symmetric key \n"));
        return DX_FAILURE;  
      }
      TST_tempInd++;
      /* Export RSA key */
      TST_RingListRSA_TestData[TST_index].TST_ExportedNSize = TST_KMNG_RSA_2048_KEYSIZE*TST_KMNG_BITS_IN_BYTE;
      TST_RingListRSA_TestData[TST_index].TST_ExportedDSize = TST_KMNG_RSA_2048_KEYSIZE*TST_KMNG_BITS_IN_BYTE;
      TST_RingListRSA_TestData[TST_index].TST_ExportedESize = TST_KMNG_BITS_IN_BYTE*4;
      	  
      TST_Status = KMNG_ExportRSAUserKey(TST_KeyRingPwd, 
                                         TST_KeyRingPwdLen,
                                         TST_RingListRSA_TestData[TST_index].TST_KeyPwd , 
                                         TST_RingListRSA_TestData[TST_index].TST_KeyPwdLen,
                                         TST_KeyIds[TST_tempInd],
                                         TST_RingListRSA_TestData[TST_index].TST_ExportedN,
                                         &TST_RingListRSA_TestData[TST_index].TST_ExportedNSize,
                                         TST_RingListRSA_TestData[TST_index].TST_ExportedE,
                                         &TST_RingListRSA_TestData[TST_index].TST_ExportedESize,
                                         TST_RingListRSA_TestData[TST_index].TST_ExportedD,
                                         &TST_RingListRSA_TestData[TST_index].TST_ExportedDSize,
                                         (DxUint8_t*)TST_KeyRing_Buf);
                                         
     TST_ATP_CHECK_ERROR_RETURN(TST_Status,
													      "KMNG_ExportRSAUserKey",
													      TST_RingListRSA_TestData[TST_index].TST_Name,
													      TST_MasterTestName);

      /* Compare exported key to imported key - E*/
      if (TST_RingListRSA_TestData[TST_index].TST_ESize != TST_RingListRSA_TestData[TST_index].TST_ExportedESize)
      {
        ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Exported RSA key E size is different from imported \n"));
        return DX_FAILURE;    
      }
      TST_Status = (DxUint32_t)DX_VOS_MemCmp(TST_RingListRSA_TestData[TST_index].TST_E,TST_RingListRSA_TestData[TST_index].TST_ExportedE, 
                                 TST_RingListRSA_TestData[TST_index].TST_ESize);
      if (TST_Status != DX_SUCCESS)
      {
        ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Exported RSA key E is different from imported \n"));
        return DX_FAILURE;  
      }
      /* Compare exported key to imported key - N*/
      if (TST_RingListRSA_TestData[TST_index].TST_NSize != TST_RingListRSA_TestData[TST_index].TST_ExportedNSize)
      {
        ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Exported RSA key N size is different from imported \n"));
        return DX_FAILURE;    
      }
      TST_Status = (DxUint32_t)DX_VOS_MemCmp(TST_RingListRSA_TestData[TST_index].TST_N,TST_RingListRSA_TestData[TST_index].TST_ExportedN, 
                                 TST_RingListRSA_TestData[TST_index].TST_NSize);
      if (TST_Status != DX_SUCCESS)
      {
        ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Exported RSA key N is different from imported \n"));
        return DX_FAILURE;  
      }
      /* Compare exported key to imported key - D*/
      if (TST_RingListRSA_TestData[TST_index].TST_DSize != TST_RingListRSA_TestData[TST_index].TST_ExportedDSize)
      {
        ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Exported RSA key D size is different from imported \n"));
        return DX_FAILURE;    
      }
      TST_Status = (DxUint32_t)DX_VOS_MemCmp(TST_RingListRSA_TestData[TST_index].TST_D,TST_RingListRSA_TestData[TST_index].TST_ExportedD, 
                                 TST_RingListRSA_TestData[TST_index].TST_DSize);
      if (TST_Status != DX_SUCCESS)
      {
        ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Exported RSA key D is different from imported \n"));
        return DX_FAILURE;  
      }
      TST_tempInd++;      
    }/* End of export keys loop */
            
    /************ Delete Keys **********/    
    TST_tempInd = 0;
    for(TST_index = 0; TST_index < TST_NumOf_SymListTests; TST_index++)
    {
      
      /* Get num of current keys in list, get list of key IDs and compare */
      TST_Status = KMNG_TST_List_Key_IDs(TST_NumOf_SymListTests-TST_index, TST_NumOf_RSAListTests-TST_index, 
                                         0, &TST_KeyIds[TST_tempInd],0,
                                         (DxUint8_t*)TST_KeyRing_Buf, TST_KMNG_SORT_LIST_A);
    
      if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;
      
      /* Delete Sym key */	  
      TST_Status = KMNG_DeleteKey(TST_KeyRingPwd, 
                                  TST_KeyRingPwdLen,
                                  TST_RingListSym_TestData[TST_index].TST_KeyPwd , 
                                  TST_RingListSym_TestData[TST_index].TST_KeyPwdLen,
                                  TST_KeyIds[TST_tempInd],
                                  (DxUint8_t*)TST_KeyRing_Buf);
      
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                "KMNG_DeleteKey",
                                TST_RingListSym_TestData[TST_index].TST_Name,
                                TST_MasterTestName);

      
      TST_tempInd++; 
      /* Delete RSA key */     
      TST_Status = KMNG_DeleteKey(TST_KeyRingPwd, 
                                  TST_KeyRingPwdLen,
                                  TST_RingListRSA_TestData[TST_index].TST_KeyPwd , 
                                  TST_RingListRSA_TestData[TST_index].TST_KeyPwdLen,
                                  TST_KeyIds[TST_tempInd],
                                  (DxUint8_t*)TST_KeyRing_Buf);
      
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
													      "KMNG_DeleteKey",
													      TST_RingListRSA_TestData[TST_index].TST_Name,
													      TST_MasterTestName); 
      
      TST_tempInd++;      
    }
    
      return DX_SUCCESS;

EXIT_ON_ERROR:
    {
      return DX_FAILURE;
    }    
}

/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/


/*****************************************************************************
 * Function Name:                                                           
 *  KMNG_ATP_TST_Operations_2_Key_Ring      
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is an acceptance test for Key management API's. The function 
 *  tests the list key IDs function.
 *  
 * Algorithm:       
 * 1. Initialize 2 Key Rings (A & B) according to requested number of keys
 * 2. Import AES key and HMAC key in Key Ring A
 * 3. Create RSA & DH key in Key Ring A
 * 4. Import RSA key in Key Ring B
 * 5. Transfer AES key from Key Ring A to B
 * 6. List Ids and verify that both KR sizes are Ok
 * 7. Repeat for all keys in KR A
 * 8. Transfer the AES key from KR B to KR A
 * 9. List Ids and verify that both KR sizes are Ok
 * 
 *  * Implements KMNG ATP section 5.6 
 *
 ******************************************************************************/                                
DxError_t KMNG_ATP_TST_Operations_2_Key_Ring( void ) 
{

    /********* TST Variables Definitions ************/
    DxUint32_t            TST_index = 0;    
    DxError_t             TST_Status = DX_SUCCESS;        
    DxChar                TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_ATP_TST_Operations_2_Key_Ring"; 


    DxUint8_t             *TST_KeyRingA, *TST_KeyRingB;
    DxUint32_t            TST_KeyIdsKeyRingA[TST_KMNG_MAX_NUM_OF_KEYS];
    DxUint32_t            TST_KeyIdsKeyRingB[TST_KMNG_MAX_NUM_OF_KEYS];
    DxUint32_t            TST_NumOfSymKeysKRA = 0, TST_NumOfSymKeysKRB = 0, 
                          TST_NumOfRSAKeysKRA = 0, TST_NumOfRSAKeysKRB = 0, 
                          TST_NumOfDHKeysKRA = 0, TST_NumOfDHKeysKRB = 0,
                          TST_MaxNumOfSymKeys = 0;
                              
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"************** KMNG_ATP_TST_Operations_2_Key_Ring Test ************** \n"));      
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n \n"));
        
    /************ Initialize Key Ring A & B ************/
    /* Initialize Key ring buffer. A points to the start of the global buffer, B points to 
       the middle of the buffer */
    TST_KeyRingA = (DxUint8_t*)TST_KeyRing_Buf;
    TST_KeyRingB = (DxUint8_t*)&TST_KeyRing_Buf[KMNG_TST_MAX_SIZE_KEY_RING_BUF/2];
    
    TST_Status = KMNG_TST_Key_Ring_Initialization(TST_RingManage_TestData[1].TST_NumSymKeys, TST_RingManage_TestData[1].TST_NumRSAKeys, 
                                                  TST_RingManage_TestData[1].TST_NumDHKeys, TST_TwoRingsSymKey_TestData[0].TST_Pwd, 
                                                  TST_TwoRingsSymKey_TestData[0].TST_PwdLen,
                                                  TST_KeyRingA, KMNG_TST_MAX_SIZE_KEY_RING_BUF_BYTES/2);
                                           
    if (TST_Status != DX_SUCCESS)
        return DX_FAILURE; 
    
    TST_Status = KMNG_TST_Key_Ring_Initialization(TST_RingManage_TestData[1].TST_NumSymKeys, TST_RingManage_TestData[1].TST_NumRSAKeys, 
                                                  TST_RingManage_TestData[1].TST_NumDHKeys, TST_TwoRingsRSAKey_TestData[0].TST_Pwd, 
                                                  TST_TwoRingsRSAKey_TestData[0].TST_PwdLen,
                                                  TST_KeyRingB, KMNG_TST_MAX_SIZE_KEY_RING_BUF_BYTES/2);
                                           
    if (TST_Status != DX_SUCCESS)
        return DX_FAILURE; 
    
    /* Initialize max num of keys in both Key Rings */
    TST_MaxNumOfSymKeys = TST_RingManage_TestData[1].TST_NumSymKeys;
  

    /************ Key Ring A - Import and create keys ************/
    for (TST_index = 0; TST_index < 2; TST_index++)
    {
      /* Import Sym key - AES & HMAC*/    
      TST_Status = KMNG_ImportSymUserKey(TST_TwoRingsSymKey_TestData[TST_index].TST_Pwd,
                                         TST_TwoRingsSymKey_TestData[TST_index].TST_PwdLen,
                                         TST_TwoRingsSymKey_TestData[TST_index].TST_KeyPwd, 
                                         TST_TwoRingsSymKey_TestData[TST_index].TST_KeyPwdLen, 
                                         TST_TwoRingsSymKey_TestData[TST_index].TST_KeyType,
                                         TST_TwoRingsSymKey_TestData[TST_index].TST_KeySize,
                                         TST_TwoRingsSymKey_TestData[TST_index].TST_KeyUsage, 
                                         TST_TwoRingsSymKey_TestData[TST_index].TST_KeyRestriction,
                                         TST_TwoRingsSymKey_TestData[TST_index].TST_UserData,
                                         TST_TwoRingsSymKey_TestData[TST_index].TST_key,
                                         &TST_KeyIdsKeyRingA[TST_index],
                                         TST_KeyRingA); 
      
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                "KMNG_ImportSymUserKey",
                                TST_TwoRingsSymKey_TestData[TST_index].TST_Name,
                                TST_MasterTestName);
                                                 

    }
    
    TST_NumOfSymKeysKRA=TST_index;
    
    /* Create RSA key - save it from index = TST_MaxNumOfSymKeys and up*/
    TST_Status = KMNG_CreateUserKey(TST_TwoRingsRSAKey_TestData[1].TST_Pwd,
                                    TST_TwoRingsRSAKey_TestData[1].TST_PwdLen,
                                    TST_TwoRingsRSAKey_TestData[1].TST_KeyPwd, 
                                    TST_TwoRingsRSAKey_TestData[1].TST_KeyPwdLen,
                                    TST_TwoRingsRSAKey_TestData[1].TST_KeyType,
                                    TST_TwoRingsRSAKey_TestData[1].TST_KeySize,
                                    TST_TwoRingsRSAKey_TestData[1].TST_KeyUsage,
                                    TST_TwoRingsRSAKey_TestData[1].TST_KeyRestriction,
                                    TST_TwoRingsRSAKey_TestData[1].TST_UserData,
                                    &TST_KeyIdsKeyRingA[TST_MaxNumOfSymKeys],
                                    TST_KeyRingA);
    
    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                               "KMNG_CreateUserKey",
                               TST_TwoRingsRSAKey_TestData[1].TST_Name,
                               TST_MasterTestName);
                                               
  
    TST_index++;
    TST_NumOfRSAKeysKRA=TST_index-TST_NumOfSymKeysKRA;
    
    TST_NumOfDHKeysKRA=TST_index - TST_NumOfSymKeysKRA - TST_NumOfRSAKeysKRA;
    
    /************ Key Ring B - Import key ************/
    TST_index = 0;
    /* Import RSA key save it from index = TST_MaxNumOfSymKeys (save the first indexes for
       symmetric keys) */    
    TST_Status = KMNG_ImportRsaUserKey(TST_TwoRingsRSAKey_TestData[0].TST_Pwd,
                                       TST_TwoRingsRSAKey_TestData[0].TST_PwdLen,
                                       TST_TwoRingsRSAKey_TestData[0].TST_KeyPwd ,
                                       TST_TwoRingsRSAKey_TestData[0].TST_KeyPwdLen,
                                       TST_TwoRingsRSAKey_TestData[0].TST_KeyType,
                                       TST_TwoRingsRSAKey_TestData[0].TST_KeySize,
                                       TST_TwoRingsRSAKey_TestData[0].TST_KeyUsage,
                                       TST_TwoRingsRSAKey_TestData[0].TST_KeyRestriction,
                                       TST_TwoRingsRSAKey_TestData[0].TST_UserData,
                                       TST_TwoRingsRSAKey_TestData[0].TST_N,
                                       TST_TwoRingsRSAKey_TestData[0].TST_NSize,
                                       TST_TwoRingsRSAKey_TestData[0].TST_E,
                                       TST_TwoRingsRSAKey_TestData[0].TST_ESize,
                                       TST_TwoRingsRSAKey_TestData[0].TST_D,
                                       TST_TwoRingsRSAKey_TestData[0].TST_DSize,
                                       &TST_KeyIdsKeyRingB[TST_MaxNumOfSymKeys],
                                       TST_KeyRingB);
                                       
    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                               "KMNG_ImportRsaUserKey",
                               TST_TwoRingsRSAKey_TestData[0].TST_Name,
                               TST_MasterTestName);
                                                 

    TST_index++;
    TST_NumOfRSAKeysKRB=TST_index;
     
    /************ Transfer AES key from key ring A to B ************/
    TST_Status = KMNG_TransferKey(TST_TwoRingsSymKey_TestData[0].TST_Pwd,/*KRA pwd*/
                                  TST_TwoRingsSymKey_TestData[0].TST_PwdLen,
                                  TST_TwoRingsSymKey_TestData[0].TST_KeyPwd, 
                                  TST_TwoRingsSymKey_TestData[0].TST_KeyPwdLen,
                                  TST_TwoRingsSymKey_TestData[2].TST_Pwd,/*KRB pwd*/
                                  TST_TwoRingsSymKey_TestData[2].TST_PwdLen,
                                  TST_TwoRingsSymKey_TestData[2].TST_KeyPwd, 
                                  TST_TwoRingsSymKey_TestData[2].TST_KeyPwdLen,
                                  TST_KeyIdsKeyRingA[0],
                                  &TST_KeyIdsKeyRingB[0],
                                  TST_KeyRingA,
                                  TST_KeyRingB);
                                  
    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                               "KMNG_TransferKey",
                               "Transfer AES key from key ring A to B",
                               TST_MasterTestName);
                               
    
    /* Update the Key ID */
    TST_TwoRingsSymKey_TestData[2].TST_KeyID = TST_KeyIdsKeyRingB[0];
    /* Update the number of the symmetric keys in both lists */
    TST_NumOfSymKeysKRA--;
    TST_NumOfSymKeysKRB++;
      
    /************ Get Key Ring A size & List Key IDs **********/
    /* The first key in the create list was transferred to KRB */
    TST_Status = KMNG_TST_List_Key_IDs(TST_NumOfSymKeysKRA, TST_NumOfRSAKeysKRA, 
                                       TST_NumOfDHKeysKRA, TST_KeyIdsKeyRingA,1,
                                       TST_KeyRingA, TST_KMNG_SORT_LIST_C);
    
    if (TST_Status != DX_SUCCESS)
        return DX_FAILURE; 
    
    /************ Get Key Ring B size & List Key IDs **********/
    TST_Status = KMNG_TST_List_Key_IDs(TST_NumOfSymKeysKRB, TST_NumOfRSAKeysKRB, 
                                       TST_NumOfDHKeysKRB, TST_KeyIdsKeyRingB,0,
                                       TST_KeyRingB, TST_KMNG_SORT_LIST_C);
    
    if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;
    
    /************ Activate and use the AES key (Key Ring B) **********/
    TST_Status = KMNG_TST_Activate_And_Use_AES_Integrate(&TST_TwoRingsSymKey_TestData[2], 
                                                         TST_KeyRingB, TST_KMNG_VERIFY_OUTPUT);
    
    if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;                                                         
    
    /************ Transfer HMAC key from key ring A to B ************/
    TST_Status = KMNG_TransferKey(TST_TwoRingsSymKey_TestData[1].TST_Pwd,/*KRA pwd*/
                                  TST_TwoRingsSymKey_TestData[1].TST_PwdLen,
                                  TST_TwoRingsSymKey_TestData[1].TST_KeyPwd, 
                                  TST_TwoRingsSymKey_TestData[1].TST_KeyPwdLen,
                                  TST_TwoRingsSymKey_TestData[3].TST_Pwd,/*KRB pwd*/
                                  TST_TwoRingsSymKey_TestData[3].TST_PwdLen,
                                  TST_TwoRingsSymKey_TestData[3].TST_KeyPwd, 
                                  TST_TwoRingsSymKey_TestData[3].TST_KeyPwdLen,
                                  TST_KeyIdsKeyRingA[1],
                                  &TST_KeyIdsKeyRingB[TST_index],
                                  TST_KeyRingA,
                                  TST_KeyRingB);
                                  
    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                               "KMNG_TransferKey",
                               "Transfer HMAC key from key ring A to B",
                               TST_MasterTestName);
                                                 
                                
    
    /* Update the key ID */
    TST_TwoRingsSymKey_TestData[3].TST_KeyID = TST_KeyIdsKeyRingB[TST_index];
    /* Update the number of the symmetric keys in both lists */
    TST_NumOfSymKeysKRA--;
    TST_NumOfSymKeysKRB++;
    TST_index++;
      
    /************ Get Key Ring A size & List Key IDs **********/
    TST_Status = KMNG_TST_List_Key_IDs(TST_NumOfSymKeysKRA, TST_NumOfRSAKeysKRA, 
                                       TST_NumOfDHKeysKRA, TST_KeyIdsKeyRingA,2,
                                       TST_KeyRingA, TST_KMNG_SORT_LIST_C);
    
    if (TST_Status != DX_SUCCESS)
        return DX_FAILURE; 
    
    /************ Get Key Ring B size & List Key IDs **********/
    TST_Status = KMNG_TST_List_Key_IDs(TST_NumOfSymKeysKRB, TST_NumOfRSAKeysKRB, 
                                       TST_NumOfDHKeysKRB, TST_KeyIdsKeyRingB,0,
                                       TST_KeyRingB, TST_KMNG_SORT_LIST_C);
    
    if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;
    
    /************ Activate and use the AES key (Key Ring B) **********/
    TST_Status = KMNG_TST_Activate_And_Use_HMAC_Integrate(&TST_TwoRingsSymKey_TestData[3], 
                                                         TST_KeyRingB, TST_KMNG_VERIFY_OUTPUT);
    
    if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;                                                         
    
    /************ Transfer RSA key from key ring A to B ************/
    TST_Status = KMNG_TransferKey(TST_TwoRingsRSAKey_TestData[1].TST_Pwd,/*KRA pwd*/
                                  TST_TwoRingsRSAKey_TestData[1].TST_PwdLen,
                                  TST_TwoRingsRSAKey_TestData[1].TST_KeyPwd, 
                                  TST_TwoRingsRSAKey_TestData[1].TST_KeyPwdLen,
                                  TST_TwoRingsRSAKey_TestData[2].TST_Pwd,/*KRB pwd*/
                                  TST_TwoRingsRSAKey_TestData[2].TST_PwdLen,
                                  TST_TwoRingsRSAKey_TestData[2].TST_KeyPwd, 
                                  TST_TwoRingsRSAKey_TestData[2].TST_KeyPwdLen,
                                  TST_KeyIdsKeyRingA[TST_MaxNumOfSymKeys],
                                  &TST_KeyIdsKeyRingB[TST_MaxNumOfSymKeys+1],
                                  TST_KeyRingA,
                                  TST_KeyRingB);
                                  
    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                               "KMNG_TransferKey",
                               "Transfer RSA key from key ring A to B",
                               TST_MasterTestName);
                                                 
                               

    /* Update the Key ID */
    TST_TwoRingsRSAKey_TestData[2].TST_KeyID = TST_KeyIdsKeyRingB[TST_MaxNumOfSymKeys+1];
    /* Update the number of the symmetric keys in both lists */
    TST_NumOfRSAKeysKRA--;
    TST_NumOfRSAKeysKRB++;
    TST_index++;
      
    /************ Get Key Ring A size & List Key IDs **********/
    TST_Status = KMNG_TST_List_Key_IDs(TST_NumOfSymKeysKRA, TST_NumOfRSAKeysKRA, 
                                       TST_NumOfDHKeysKRA, TST_KeyIdsKeyRingA,0,
                                       TST_KeyRingA, TST_KMNG_SORT_LIST_C);
    
    if (TST_Status != DX_SUCCESS)
        return DX_FAILURE; 
    
    /************ Get Key Ring B size & List Key IDs **********/
    TST_Status = KMNG_TST_List_Key_IDs(TST_NumOfSymKeysKRB, TST_NumOfRSAKeysKRB, 
                                       TST_NumOfDHKeysKRB, TST_KeyIdsKeyRingB,0,
                                       TST_KeyRingB, TST_KMNG_SORT_LIST_C);
    
    if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;
    
    /************ Activate and use the AES key (Key Ring B) **********/
    TST_Status = KMNG_TST_Activate_And_Use_RSA_EncryptDecrypt(&TST_TwoRingsRSAKey_TestData[2], 
                                               TST_KeyRingB, TST_KMNG_VERIFY_OUTPUT);
    
    if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;
    
    /************ Transfer AES key from key ring B to A ************/
    TST_Status = KMNG_TransferKey(TST_TwoRingsSymKey_TestData[2].TST_Pwd,/*KRA pwd*/
                                  TST_TwoRingsSymKey_TestData[2].TST_PwdLen,
                                  TST_TwoRingsSymKey_TestData[2].TST_KeyPwd, 
                                  TST_TwoRingsSymKey_TestData[2].TST_KeyPwdLen,
                                  TST_TwoRingsSymKey_TestData[0].TST_Pwd,/*KRB pwd*/
                                  TST_TwoRingsSymKey_TestData[0].TST_PwdLen,
                                  TST_TwoRingsSymKey_TestData[0].TST_KeyPwd, 
                                  TST_TwoRingsSymKey_TestData[0].TST_KeyPwdLen,
                                  TST_KeyIdsKeyRingB[0],
                                  &TST_KeyIdsKeyRingA[1],
                                  TST_KeyRingB,
                                  TST_KeyRingA);
                                  
    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                               "KMNG_TransferKey",
                               "Transfer AES key from key ring B to A",
                               TST_MasterTestName);
                                                 
                                  

    /* Update the Key ID */
    TST_TwoRingsSymKey_TestData[0].TST_KeyID = TST_KeyIdsKeyRingA[1];
    /* Update the number of the symmetric keys in both lists */
    TST_NumOfSymKeysKRB--;
    TST_NumOfSymKeysKRA++;
      
    /************ Get Key Ring A size & List Key IDs **********/
    TST_Status = KMNG_TST_List_Key_IDs(TST_NumOfSymKeysKRA, TST_NumOfRSAKeysKRA, 
                                       TST_NumOfDHKeysKRA, TST_KeyIdsKeyRingA,1,
                                       TST_KeyRingA, TST_KMNG_SORT_LIST_C);
    
    if (TST_Status != DX_SUCCESS)
        return DX_FAILURE; 
    
    /************ Get Key Ring B size & List Key IDs **********/
    TST_Status = KMNG_TST_List_Key_IDs(TST_NumOfSymKeysKRB, TST_NumOfRSAKeysKRB, 
                                       TST_NumOfDHKeysKRB, TST_KeyIdsKeyRingB,1,
                                       TST_KeyRingB, TST_KMNG_SORT_LIST_C);
    
    if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;
    
    /************ Activate and use the AES key (Key Ring A) **********/
    TST_Status = KMNG_TST_Activate_And_Use_AES_Integrate(&TST_TwoRingsSymKey_TestData[0], 
                                                         TST_KeyRingA, TST_KMNG_VERIFY_OUTPUT);
    
    if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;
    
   return DX_SUCCESS;

EXIT_ON_ERROR:
    {
      return DX_FAILURE;
    } 
}


/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/


/*****************************************************************************
 * Function Name:                                                           
 *  KMNG_ATP_TST_Save_Key_Ring_To_Flash      
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is an acceptance test for Key management API's. The function 
 *  tests the list key IDs function.
 *  
 * Algorithm:       
 * 1. Initialize a key ring according to requested number of keys
 * 2. Create Symmetric,RSA and DH keys
 * 3. Get the flash needed size for saving the Key Ring
 * 4. Save the key ring to "flash" (buffer)
 * 5. Get Key Ring data from flash image
 * 6. Restore key ring from image
 * 7. Get key ring size and list Ids to verify its as expected
 * 
 *  * Implements KMNG ATP section 5.7 
 *
 ******************************************************************************/                                
DxError_t KMNG_ATP_TST_Save_Key_Ring_To_Flash( void ) 
{

    /********* TST Variables Definitions ************/
    DxUint32_t            TST_index, TST_tempInd=0;    
    DxError_t             TST_Status = DX_SUCCESS;        
    DxChar                TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_ATP_TST_Save_Key_Ring_To_Flash"; 
    DxUint8_t             TST_KeyRingPwd[TST_KMNG_PWD_MAX_LEN] = {0x0,0x1,0x2,0x3,0x4,0x5,0x6,0x7,0x8,0x9};
    DxUint32_t            TST_KeyRingPwdLen = 10;
    DxUint32_t            TST_KeyIds[TST_KMNG_MAX_NUM_OF_KEYS];               
    DxUint32_t            TST_KeyRingSizeForFlash = 0;
    DxUint32_t            TST_NumOfSymKeys = 0, TST_NumOfRSAKeys = 0, TST_NumOfDHKeys = 0;
                              
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"************** KMNG_ATP_TST_Save_Key_Ring_To_Flash Test ********************** \n"));      
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n \n"));
    
    /************ Initialize Key Ring ************/
    TST_Status = KMNG_TST_Key_Ring_Initialization(TST_RingManage_TestData[1].TST_NumSymKeys, TST_RingManage_TestData[1].TST_NumRSAKeys, 
                                                  TST_RingManage_TestData[1].TST_NumDHKeys, TST_KeyRingPwd, 
                                                  TST_KeyRingPwdLen,(DxUint8_t*)TST_KeyRing_Buf, KMNG_TST_MAX_SIZE_KEY_RING_BUF_BYTES);
                                           
    if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;                                       
                                           
    /************ Create keys Loop **********/
    /* the loop is according to the sym keys,in this test 
       number of sym keys = num of RSA keys */    
    for (TST_index = 0; TST_index < TST_NumOf_SymListTests; TST_index++)
    {      
      /* Create Sym key */
      TST_Status = KMNG_CreateUserKey(TST_KeyRingPwd,
                                      TST_KeyRingPwdLen,
                                      TST_RingListSym_TestData[TST_index].TST_KeyPwd, 
                                      TST_RingListSym_TestData[TST_index].TST_KeyPwdLen, 
                                      TST_RingListSym_TestData[TST_index].TST_KeyType,
                                      TST_RingListSym_TestData[TST_index].TST_KeySize,
                                      TST_RingListSym_TestData[TST_index].TST_KeyUsage, 
                                      TST_RingListSym_TestData[TST_index].TST_KeyRestriction,
                                      DX_NULL,
                                      &TST_KeyIds[TST_tempInd],
                                      (DxUint8_t*)TST_KeyRing_Buf); 
      
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
													      "KMNG_CreateUserKey",
													      TST_RingListSym_TestData[TST_index].TST_Name,
													      TST_MasterTestName);

      
      TST_tempInd++;
      
      /* Create RSA key */
      TST_Status = KMNG_CreateUserKey(TST_KeyRingPwd, 
                                      TST_KeyRingPwdLen,
                                      TST_RingListRSA_TestData[TST_index].TST_KeyPwd ,
                                      TST_RingListRSA_TestData[TST_index].TST_KeyPwdLen,
                                      TST_RingListRSA_TestData[TST_index].TST_KeyType,
                                      TST_RingListRSA_TestData[TST_index].TST_KeySize,
                                      TST_RingListRSA_TestData[TST_index].TST_KeyUsage,
                                      TST_RingListRSA_TestData[TST_index].TST_KeyRestriction,
                                      DX_NULL,
                                      &TST_KeyIds[TST_tempInd],
                                      (DxUint8_t*)TST_KeyRing_Buf);
                                         
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                                "KMNG_CreateUserKey",
                                TST_RingListRSA_TestData[TST_index].TST_Name,
                                TST_MasterTestName);
                                                 
      
      TST_tempInd++;     
                                              
    } /*End of for loop*/
    
    /************ Get flash required buffer size **********/
    TST_Status = KMNG_GetFlashBufferSize((DxUint8_t*)TST_KeyRing_Buf,
                                         &TST_KeyRingSizeForFlash);
                                
    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                               "KMNG_GetFlashBufferSize",
                               "Get flash buffer size",
                                TST_MasterTestName);
                                                 
 
    
    if (TST_KeyRingSizeForFlash == 0 || TST_KeyRingSizeForFlash > KMNG_TST_MAX_SIZE_KEY_RING_BUF_BYTES)
    {
      ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Flash required size - %d is illegal\n",TST_KeyRingSizeForFlash));   
      return DX_FAILURE;  
    }
    
    /************ Save Key Ring to buffer **********/
    TST_Status = KMNG_SaveKeyRingToBuffer((DxUint8_t*)TST_KeyRing_Buf,
                                          (DxUint8_t*)TST_KeyRing_FlashBuf,
                                          TST_KeyRingSizeForFlash);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                               "KMNG_SaveKeyRingToBuffer",
                               "Save Key Ring to flash",
                               TST_MasterTestName);
                                                 

    
    TST_Status = KMNG_GetKeyRingDataFromImage((DxUint8_t*)TST_KeyRing_FlashBuf,
                                              TST_KeyRingSizeForFlash,
                                              &TST_NumOfSymKeys,
                                              &TST_NumOfRSAKeys,
                                              &TST_NumOfDHKeys);
                                              
    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                               "KMNG_GetKeyRingDataFromImage",
                               "Get Key Ring data from image",
                               TST_MasterTestName);
                                                 

    
    DX_VOS_MemSetZero((DxUint8_t*)TST_KeyRing_Buf, KMNG_TST_MAX_SIZE_KEY_RING_BUF*sizeof(DxUint32_t));

    /************ Restore Key Ring from flash **********/
    TST_Status = KMNG_RestoreKeyRingFromImage((DxUint8_t*)TST_KeyRing_FlashBuf,
                                              TST_KeyRingSizeForFlash,
                                              (DxUint8_t*)TST_KeyRing_Buf,
                                              KMNG_TST_MAX_SIZE_KEY_RING_BUF_BYTES);
                                              
    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                               "KMNG_RestoreKeyRingFromImage",
                               "Restore Key Ring from flash",
                                TST_MasterTestName);
                                                 
                                              
    
    /************ Get Key Ring size & List Key IDs **********/
    TST_Status = KMNG_TST_List_Key_IDs(TST_NumOfSymKeys, TST_NumOfRSAKeys, 
                                       0, TST_KeyIds,0,
                                       (DxUint8_t*)TST_KeyRing_Buf, TST_KMNG_SORT_LIST_A);
    
    if (TST_Status != DX_SUCCESS)
        return DX_FAILURE;
    
    return DX_SUCCESS;
    EXIT_ON_ERROR:
    {
      return DX_FAILURE;
    } 
}


/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/

/*****************************************************************************
* Function Name:                                                           
*  KMNG_ATP_TST_Get_Keys_Data      
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is an acceptance test for Key management API's. The function 
*  initializes a key ring according to requested num of keys
*  
* Algorithm:       
* 1. Initialize Key Ring
* 2. Create RSA key and import symmetric key 
* 3. Retrieve the data from the keys and verify its correct
* 4. Update the keys user data
* 5. Retrieve the data from the keys and verify its correct
*
*  * Implements KMNG ATP section 5.8 
*
******************************************************************************/                                
DxError_t KMNG_ATP_TST_Get_Keys_Data( void ) 
{
  /********* TST Variables Definitions ************/
  DxUint32_t                  TST_index;    
  DxError_t                   TST_Status = DX_SUCCESS;        
  DxChar                      TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_ATP_TST_Get_Keys_Data";   
  KMNG_UserSpecificKeyData_t  TST_UpdateUserData = {0xaaaa, 0xbbbb};
  
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n"));
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"******** KMNG_ATP_TST_Get_Keys_Data Test* ********************* \n"));      
  ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"********************************************************************* \n \n"));

  /************ Initialize Key Ring ************/
  TST_Status = KMNG_TST_Key_Ring_Initialization(TST_RingManage_TestData[0].TST_NumSymKeys, TST_RingManage_TestData[0].TST_NumRSAKeys, 
                                                TST_RingManage_TestData[0].TST_NumDHKeys, TST_KeysSymData_TestData[0].TST_Pwd, 
                                                TST_KeysSymData_TestData[0].TST_PwdLen,(DxUint8_t*)TST_KeyRing_Buf, KMNG_TST_MAX_SIZE_KEY_RING_BUF_BYTES);

  if (TST_Status != DX_SUCCESS)
    return DX_FAILURE; 

  /************ Main Loop **********/
  for (TST_index = 0; TST_index < TST_NumOf_KeysRSADataTests; TST_index++)
  {
    /******** Create and import keys **********/
    /* Import Sym key - AES */    
    TST_Status = KMNG_ImportSymUserKey(TST_KeysSymData_TestData[TST_index].TST_Pwd,
                                       TST_KeysSymData_TestData[TST_index].TST_PwdLen,
                                       TST_KeysSymData_TestData[TST_index].TST_KeyPwd, 
                                       TST_KeysSymData_TestData[TST_index].TST_KeyPwdLen, 
                                       TST_KeysSymData_TestData[TST_index].TST_KeyType,
                                       TST_KeysSymData_TestData[TST_index].TST_KeySize,
                                       TST_KeysSymData_TestData[TST_index].TST_KeyUsage, 
                                       TST_KeysSymData_TestData[TST_index].TST_KeyRestriction,
                                       TST_KeysSymData_TestData[TST_index].TST_UserData,
                                       TST_KeysSymData_TestData[TST_index].TST_key,
                                       &TST_KeysSymData_TestData[TST_index].TST_KeyID,
                                       (DxUint8_t*)TST_KeyRing_Buf); 

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                              "KMNG_ImportSymUserKey",
                              TST_KeysSymData_TestData[TST_index].TST_Name,
                              TST_MasterTestName);


    /* Create RSA key */
    TST_Status = KMNG_CreateUserKey(TST_KeysRSAData_TestData[TST_index].TST_Pwd,
                                    TST_KeysRSAData_TestData[TST_index].TST_PwdLen,
                                    TST_KeysRSAData_TestData[TST_index].TST_KeyPwd, 
                                    TST_KeysRSAData_TestData[TST_index].TST_KeyPwdLen,
                                    TST_KeysRSAData_TestData[TST_index].TST_KeyType,
                                    TST_KeysRSAData_TestData[TST_index].TST_KeySize,
                                    TST_KeysRSAData_TestData[TST_index].TST_KeyUsage,
                                    TST_KeysRSAData_TestData[TST_index].TST_KeyRestriction,
                                    TST_KeysRSAData_TestData[TST_index].TST_UserData,
                                    &TST_KeysRSAData_TestData[TST_index].TST_KeyID,
                                    (DxUint8_t*)TST_KeyRing_Buf);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
												      "KMNG_CreateUserKey",
												      TST_KeysRSAData_TestData[TST_index].TST_Name,
												      TST_MasterTestName);


    /******** Retrieve the data from the keys and verify its correct **********/
    /* Retrieve symmetric key data */
    TST_Status = KMNG_TST_Retrieve_KeysData_And_Compare(DX_NULL, &TST_KeysSymData_TestData[TST_index]);
    if (TST_Status != KMNG_TEST_PASS)
      return DX_FAILURE;
    
    /* Retrieve RSA key data */
    TST_Status = KMNG_TST_Retrieve_KeysData_And_Compare(&TST_KeysRSAData_TestData[TST_index], DX_NULL);
    if (TST_Status != KMNG_TEST_PASS)
      return DX_FAILURE;
    
    /******** Update the keys user data **********/
    /* Update the symmetric user data */
    TST_Status = KMNG_UpdateUserSpecificKeyData(TST_KeysSymData_TestData[TST_index].TST_Pwd,
                                                TST_KeysSymData_TestData[TST_index].TST_PwdLen,
                                                TST_KeysSymData_TestData[TST_index].TST_KeyPwd,
                                                TST_KeysSymData_TestData[TST_index].TST_KeyPwdLen,
                                                TST_KeysSymData_TestData[TST_index].TST_KeyID,
                                                TST_UpdateUserData, (DxUint8_t*)TST_KeyRing_Buf);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
													    "KMNG_UpdateUserSpecificKeyData",
													     TST_KeysSymData_TestData[TST_index].TST_Name,
													     TST_MasterTestName);

  
    DX_VOS_FastMemCpy((DxUint8_t*)TST_KeysSymData_TestData[TST_index].TST_UserData, (DxUint8_t*)TST_UpdateUserData, KMNG_USER_SPECIFIC_KEY_DATA_SIZE);
    
    /* Update the RSA User data */
    TST_Status = KMNG_UpdateUserSpecificKeyData(TST_KeysRSAData_TestData[TST_index].TST_Pwd,
                                                TST_KeysRSAData_TestData[TST_index].TST_PwdLen,
                                                TST_KeysRSAData_TestData[TST_index].TST_KeyPwd,
                                                TST_KeysRSAData_TestData[TST_index].TST_KeyPwdLen,
                                                TST_KeysRSAData_TestData[TST_index].TST_KeyID,
                                                TST_UpdateUserData, (DxUint8_t*)TST_KeyRing_Buf);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
													    "KMNG_UpdateUserSpecificKeyData",
													    TST_KeysRSAData_TestData[TST_index].TST_Name,
													    TST_MasterTestName);


    DX_VOS_FastMemCpy((DxUint8_t*)TST_KeysRSAData_TestData[TST_index].TST_UserData, (DxUint8_t*)TST_UpdateUserData, KMNG_USER_SPECIFIC_KEY_DATA_SIZE);

    /******** Retrieve the data from the keys and verify its correct **********/
    /* Retrieve symmetric key data */
    TST_Status = KMNG_TST_Retrieve_KeysData_And_Compare(DX_NULL, &TST_KeysSymData_TestData[TST_index]);
    if (TST_Status != KMNG_TEST_PASS)
      return DX_FAILURE;

    /* Retrieve RSA key data */
    TST_Status = KMNG_TST_Retrieve_KeysData_And_Compare(&TST_KeysRSAData_TestData[TST_index], DX_NULL);
    if (TST_Status != KMNG_TEST_PASS)
      return DX_FAILURE;
  }

    return DX_SUCCESS;

EXIT_ON_ERROR:
    {
      return DX_FAILURE;
    } 
}/* End of KMNG_ATP_TST_Get_Keys_Data*/ 


/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/

/*****************************************************************************
* Function Name:                                                           
*  KMNG_TST_Retrieve_KeysData_And_Compare      
*  
* Inputs:
*  TST_KeysRSA_TestDataStruct *RSAKey - The RSA structure key (containing the key data)
*  TST_KeysSym_TestDataStruct *SYMKey - The symmetric structure key (containing the key data)
*
* Outputs: 
*  DxError_t - Function error return                                     
*
* Description:                                                                 
*  This function retrieves key data (according to given KeyID) and verifies it
*  
* Algorithm:       
* 1. Retrieve either the Symmetric key data or the RSA key data
* 2. Compare the key info with expected info
*
******************************************************************************/
DxError_t KMNG_TST_Retrieve_KeysData_And_Compare(TST_KeysRSA_TestDataStruct *RSAKey, TST_KeysSym_TestDataStruct *SYMKey)
{
  DxError_t                   TST_Status = DX_SUCCESS;
  KMNG_KeyType_t              TST_ReturnedKeyType;
  DxUint32_t                  TST_ReturnedKeySize = 0;
  DxUint32_t                  TST_ReturnedKeyUsage = 0;
  DxUint32_t                  TST_ReturnedKeyRestriction = 0;
  KMNG_UserSpecificKeyData_t  TST_ReturnedUserData = {0};
  DxChar                      TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_TST_Retrieve_KeysData_And_Compare";   

  /* Retrieve either the Symmetric key data or the RSA key data */
  if (RSAKey != DX_NULL)
  {    	
	TST_Status = KMNG_RetrieveUserKeyInform(RSAKey->TST_KeyID, &TST_ReturnedKeyType,
                                            &TST_ReturnedKeySize, &TST_ReturnedKeyUsage, &TST_ReturnedKeyRestriction,
                                            TST_ReturnedUserData, (DxUint8_t*)TST_KeyRing_Buf);
    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
													    "KMNG_RetrieveUserKeyInform",
													    RSAKey->TST_Name,
													    TST_MasterTestName);

    /* Compare the key info with expected info */
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"TST_message - Comparing RSA key info to expected info \n"));
    if (TST_ReturnedKeyType != RSAKey->TST_KeyType)
    {
      ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Key Type is different than expected \n"));
      return DX_FAILURE;
    }
    if (TST_ReturnedKeySize != RSAKey->TST_KeySize)
    {
      ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Key size is different than expected \n"));
      return DX_FAILURE;
    }
    if (TST_ReturnedKeyUsage != RSAKey->TST_KeyUsage)
    {
      ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Key Usage is different than expected \n"));
      return DX_FAILURE;
    }
    if (TST_ReturnedKeyRestriction != RSAKey->TST_KeyRestriction)
    {
      ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Key Restriction is different than expected \n"));
      return DX_FAILURE;
    }
    TST_Status = (DxError_t)DX_VOS_MemCmp((DxUint8_t*)TST_ReturnedUserData, (DxUint8_t*)RSAKey->TST_UserData,KMNG_USER_SPECIFIC_KEY_DATA_SIZE);
    if (TST_Status != DX_SUCCESS)
    {
      ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Key User data is different than expected \n"));
      return DX_FAILURE;
    }
  }
  if (SYMKey != DX_NULL)
  {
    TST_Status = KMNG_RetrieveUserKeyInform(SYMKey->TST_KeyID, &TST_ReturnedKeyType,
                                            &TST_ReturnedKeySize, &TST_ReturnedKeyUsage, &TST_ReturnedKeyRestriction,
                                            TST_ReturnedUserData, (DxUint8_t*)TST_KeyRing_Buf);
    TST_ATP_CHECK_ERROR_RETURN(TST_Status,
													    "KMNG_RetrieveUserKeyInform",
													    SYMKey->TST_Name,
													    TST_MasterTestName);

    /* Compare the key info with expected info */
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"TST_message - Comparing Symmetric key info to expected info \n"));
    if (TST_ReturnedKeyType != SYMKey->TST_KeyType)
    {
      ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Key Type is different than expected \n"));
      return DX_FAILURE;
    }
    if (TST_ReturnedKeySize != SYMKey->TST_KeySize)
    {
      ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Key size is different than expected \n"));
      return DX_FAILURE;
    }
    if (TST_ReturnedKeyUsage != SYMKey->TST_KeyUsage)
    {
      ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Key Usage is different than expected \n"));
      return DX_FAILURE;
    }
    if (TST_ReturnedKeyRestriction != SYMKey->TST_KeyRestriction)
    {
      ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Key Restriction is different than expected \n"));
      return DX_FAILURE;
    }
    TST_Status = (DxError_t)DX_VOS_MemCmp((DxUint8_t*)TST_ReturnedUserData, (DxUint8_t*)SYMKey->TST_UserData,KMNG_USER_SPECIFIC_KEY_DATA_SIZE);
    if (TST_Status != DX_SUCCESS)
    {
      ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Key User data is different than expected \n"));
      return DX_FAILURE;
    }
  }

      return DX_SUCCESS;

EXIT_ON_ERROR:
    {
      return DX_FAILURE;
    } 
}

/*****************************************************************************
 * Function Name:                                                           
 *  KMNG_TST_Key_Ring_Initialization      
 *  
 * Inputs:
 *  DxUint32_t NumOfSymKeys - Number of symmetric keys
 *  DxUint32_t NumOfRSAKeys - Number of RSA keys
 *  DxUint32_t NumOfDHKeys - Number of DH keys
 *  DxUint8_t* KeyRingPwd - the requested password for the initialized key ring
 *  DxUint32_t PwdLen - password length
 *  DxUint8_t* KeyRing_ptr - a pointer for the initialized key ring
 *  DxUint32_t KeyRingBufferSize - the allocated max buffer size
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function initialize the key ring to a requested number of keys
 *  
 * Algorithm:       
 * 1. Get key ring buffer size for requested number of keys (symmetric, RSA and DH)
 * 2. Verify that the requested size is allowed (smaller or equal to max size)
 * 3. Initialize the Key Ring 
 *
 ******************************************************************************/                                
DxError_t KMNG_TST_Key_Ring_Initialization(DxUint32_t NumOfSymKeys, DxUint32_t NumOfRSAKeys, 
                                           DxUint32_t NumOfDHKeys, DxUint8_t* KeyRingPwd, 
                                           DxUint32_t PwdLen, DxUint8_t* KeyRing_ptr, 
                                           DxUint32_t KeyRingBufferSize) 
{
  /********* TST Variables Definitions ************/
  DxUint32_t            TST_RequiredBufSize = 0;
  DxError_t             TST_Status = DX_SUCCESS;
  
  DX_VOS_MemSetZero((DxUint8_t*)KeyRing_ptr,KeyRingBufferSize);
  
  /* Get key ring expected buffer size */
  TST_Status = KMNG_GetKeyRingBufferSize(NumOfSymKeys, 
                                        NumOfRSAKeys,
                                        NumOfDHKeys,
                                        &TST_RequiredBufSize); 

  TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                             "KMNG_GetKeyRingBufferSize",
                             "Key Ring Initialize",
                             "KMNG_TST_Key_Ring_Initialization");
                                             

  
  if (TST_RequiredBufSize > KeyRingBufferSize)
  {
    ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Requested size - %d is bigger than pre-allocated buffer\n",TST_RequiredBufSize));   
    return DX_FAILURE;  
  }
  
  /* Initialize Key Ring */
  TST_Status = KMNG_KeyRingInit(PwdLen,
                                KeyRingPwd,
                                NumOfSymKeys,
                                NumOfRSAKeys,
                                NumOfDHKeys, 
                                (DxUint8_t*)KeyRing_ptr);
                                
  TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                             "KMNG_KeyRingInit",
                             "Key Ring Initialize",
                             "KMNG_TST_Key_Ring_Initialization");
                                             

  
        return DX_SUCCESS;

EXIT_ON_ERROR:
    {
      return DX_FAILURE;
    } 
}

/*****************************************************************************
 * Function Name:                                                           
 *  KMNG_TST_List_Key_IDs      
 *  
 * Inputs:
 *  DxUint32_t NumOfSymKeys - Number of symmetric keys
 *  DxUint32_t NumOfRSAKeys - Number of RSA keys
 *  DxUint32_t NumOfDHKeys - Number of DH keys
 *  DxUint32_t* KeyIdsList - the reference list of IDs (to compare with)
 *  DxUint8_t ListOffset  - Offset from beginning of key IDs list
 *  DxUint8_t* KeyRing_ptr - a pointer to the key ring
 *  DxUint8_t sortList - the method according to which the key list is arranged 
 *                       (TST_KMNG_SORT_LIST_A - sym key->rsa key->sym key...,
 *                        TST_KMNG_SORT_LIST_B - one after the other without specific order
 *                        TST_KMNG_SORT_LIST_C - All sym keys -> all RSA keys
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function 
 *  
 * Algorithm:       
 * 1. Get key ring size, verify its as expected
 * 2. List key ring key IDs
 * 3. Check the retrieved IDs with a given IDs list 
 *
 ******************************************************************************/
DxError_t KMNG_TST_List_Key_IDs(DxUint32_t NumOfSymKeys, DxUint32_t NumOfRSAKeys, 
                                DxUint32_t NumOfDHKeys, DxUint32_t* KeyIdsList,
                                DxUint8_t ListOffset, DxUint8_t* KeyRing_ptr,
                                DxUint8_t sortList)
{
  
  /********* TST Variables Definitions ************/
  DxError_t             TST_Status = DX_SUCCESS;
  DxUint32_t            TST_ReceivedNumSymKeys;
  DxUint32_t            TST_ReceivedNumRSAKeys;
  DxUint32_t            TST_ReceivedNumDHKeys;   
  DxChar                TST_MasterTestName[TST_FUNC_STRING_LEN] = "KMNG_TST_List_Key_IDs";
  DxUint32_t            TST_ReceivedKeyIds[TST_KMNG_MAX_NUM_OF_KEYS];
  DxUint32_t            TST_index = 0, TST_tempInd = 0;
  

  /************ Get Key Ring size *************/
  /* Get the Key Ring current size */
  TST_Status = KMNG_GetKeyRingSize(&TST_ReceivedNumSymKeys, 
                                   &TST_ReceivedNumRSAKeys,
                                   &TST_ReceivedNumDHKeys,
                                   (DxUint8_t*)KeyRing_ptr);
  
  TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                             "KMNG_GetKeyRingSize",
                             "Get Key Ring size",
                             TST_MasterTestName);
                                             

  
  /* Check if received number of keys is as expected */
  TST_Status = KMNG_TST_Check_Num_Of_Keys(TST_ReceivedNumSymKeys, NumOfSymKeys,
                                          TST_ReceivedNumRSAKeys, NumOfRSAKeys,
                                          TST_ReceivedNumDHKeys, NumOfDHKeys);
  if (TST_Status != DX_SUCCESS)
    return DX_FAILURE;
  
  /************ List the keys IDs *************/    
  TST_Status = KMNG_ListKeyIDs(&TST_ReceivedKeyIds[0], 
                               &TST_ReceivedKeyIds[NumOfSymKeys],
                               &TST_ReceivedKeyIds[NumOfSymKeys+NumOfRSAKeys],
                               (DxUint8_t*)KeyRing_ptr);
  
  TST_ATP_CHECK_ERROR_RETURN(TST_Status,
                             "KMNG_ListKeyIDs",
                             "List key IDs of the Key Ring",
                             TST_MasterTestName);
                                             

  /* If the list is sorted (the keys were inserted according to sort)
     according to symmetric -> RSA key */
  if (sortList == TST_KMNG_SORT_LIST_A)
  {
    /* Compare the received IDs with the expected IDs */
    for (TST_index = 0; TST_index < NumOfSymKeys; TST_index++)
    {
      TST_tempInd=TST_index*(TST_NumOf_SymListTests-1);
      /* Compare symmetric key Ids*/
      if (TST_ReceivedKeyIds[TST_index] != KeyIdsList[TST_tempInd])
      {
        ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Symmetric key IDs are different than expected \n"));
        return DX_FAILURE;
      }
      /* Compare RSA key Ids*/
      if (TST_ReceivedKeyIds[TST_index+NumOfSymKeys] != KeyIdsList[TST_tempInd+1])
      {
        ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"RSA key IDs are different than expected \n"));
        return DX_FAILURE;
      }      
    }
  }
  if (sortList == TST_KMNG_SORT_LIST_B)
  {
    /* Compare the received IDs with the expected IDs */
    for (TST_index = 0; TST_index < (NumOfSymKeys+NumOfRSAKeys+NumOfDHKeys); TST_index++)
    {       
      if (TST_ReceivedKeyIds[TST_index] != KeyIdsList[TST_index])
      {
        ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Key IDs are different than expected \n"));
        return DX_FAILURE;
      }            
    } 
  }
  if (sortList == TST_KMNG_SORT_LIST_C)
  {
    /* Compare the received Sym IDs with the expected IDs */
    for (TST_index = 0; TST_index < NumOfSymKeys; TST_index++)
    {       
      if (TST_ReceivedKeyIds[TST_index] != KeyIdsList[TST_index+ListOffset])
      {
        ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"Symmetric Key IDs are different than expected \n"));
        return DX_FAILURE;
      }            
    }
    /* Compare the received RSA IDs with the expected IDs */
    for (TST_index = 0; TST_index < NumOfRSAKeys; TST_index++)
    {       
      if (TST_ReceivedKeyIds[NumOfSymKeys+TST_index] != KeyIdsList[TST_NumOf_SymListTests+TST_index])
      {
        ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"RSA Key IDs are different than expected \n"));
        return DX_FAILURE;
      }            
    }
  }
   return DX_SUCCESS;

EXIT_ON_ERROR:
    {
      return DX_FAILURE;
    } 
}






